<?php
// uebergabe_druck.php – druckfertiges Übergabeprotokoll
ob_start();
session_start();

require_once __DIR__ . '/includes/db.php';

// Zugriffsschutz
if (!isset($_SESSION['user_id'])) { header('Location: login.php'); exit; }
$user_einheit = $_SESSION['einheit'] ?? '';
if ($user_einheit === '') { http_response_code(403); echo "Einheit fehlt."; exit; }

// Eingaben
$typ = $_GET['typ'] ?? '';
$id  = (int)($_GET['id'] ?? 0);

// Nur erlaubte Typen
$allowed = ['Transponder','Schluessel','Meldeempfaenger','Meldeempfänger','Melder'];
if (!in_array($typ, $allowed, true)) { http_response_code(400); echo "Ungültiger Typ."; exit; }

// Datensatz laden
$stmt = $conn->prepare("
  SELECT ts.*
  FROM transponder_schluessel ts
  WHERE ts.id = ? AND ts.einheit = ? AND ts.typ = ?
  LIMIT 1
");
$stmt->bind_param('iss', $id, $user_einheit, $typ);
$stmt->execute();
$dat = $stmt->get_result()->fetch_assoc();
if (!$dat) { http_response_code(404); echo "Datensatz nicht gefunden."; exit; }

// Kamerad via Spindnr lookup
$kStmt = $conn->prepare("
  SELECT vorname, nachname, spindnr
  FROM kameraden
  WHERE einheit = ? AND spindnr = ?
  LIMIT 1
");
$spind = (string)($dat['spindnr'] ?? '');
$kStmt->bind_param('ss', $user_einheit, $spind);
$kStmt->execute();
$k = $kStmt->get_result()->fetch_assoc();

function h($s){ return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8'); }

// Labels je Typ
$labels = [
  'Transponder'       => ['title' => 'Übergabeprotokoll – Transponder', 'field' => 'Transponder-Nummer / Kennung'],
  'Schluessel'        => ['title' => 'Übergabeprotokoll – Schlüssel',   'field' => 'Schlüsselnummer'],
  'Meldeempfaenger'   => ['title' => 'Übergabeprotokoll – Meldeempfänger', 'field' => 'Melder-Nummer'],
  'Meldeempfänger'    => ['title' => 'Übergabeprotokoll – Meldeempfänger', 'field' => 'Melder-Nummer'],
  'Melder'            => ['title' => 'Übergabeprotokoll – Meldeempfänger', 'field' => 'Melder-Nummer'],
];
$L = $labels[$typ] ?? ['title'=>'Übergabeprotokoll', 'field'=>'Nummer'];

// Übergabetext (Transponder inkl. 10€-Gebühr)
if ($typ === 'Transponder') {
  $UE_TEXT = <<<HTML
  <h5>Bedingungen für die Überlassung</h5>
  <ol>
    <li><strong>Transponder Verwahrung</strong>
      <ol>
        <li>Der/die Empfänger/In des Transponders ist für eine sichere Aufbewahrung verantwortlich. Er/Sie übernimmt die Haftung für den erhaltenen Transponder und trägt sämtliche anfallenden Kosten, die sich aus einem Verlust des Transponders ergeben.</li>
        <li>Jegliche Weitergabe des Transponders an Dritte ist untersagt; dies kann zum Einzug des Transponders führen.</li>
        <li>Der Transponder darf nicht mit in den Urlaub genommen werden.</li>
        <li>Pro Person wird nur ein Transponder ausgegeben.</li>
        <li>Für die Ausgabe wird eine Gebühr in Höhe von <strong>10,00&nbsp;€</strong> pro Transponder erhoben.</li>
      </ol>
    </li>
    <li><strong>Rückgabe des Transponders</strong>
      <ol>
        <li>Beim Ausscheiden aus der Freiwilligen Feuerwehr Bad Zwischenahn ist der erhaltene Transponder unverzüglich (binnen drei Tagen) an den Verwalter zurückzugeben.</li>
        <li>Sollte der Transponder nicht zurückgegeben werden, werden die entstandenen Kosten für die Wiederherstellung der Sicherheit dem/der Transponder-Inhaber/In in Rechnung gestellt.</li>
      </ol>
    </li>
  </ol>
  HTML;
} else {
  $UE_TEXT = "<p>Hiermit wird die Übergabe des oben genannten Objekts bestätigt. "
           . "Der Empfänger verpflichtet sich zur sorgfältigen Aufbewahrung, zweckgebundenen Nutzung sowie "
           . "zur unverzüglichen Meldung von Verlust/Beschädigung und zur Rückgabe auf Aufforderung oder bei Ausscheiden.</p>";
}

// Heutiges Datum
$heute = (new DateTime('now'))->format('d.m.Y');

// Rück-Link
$backMap = [
  'Transponder'     => 'transponder.php',
  'Schluessel'      => 'schluessel.php',
  'Meldeempfaenger' => 'melder.php',
  'Meldeempfänger'  => 'melder.php',
  'Melder'          => 'melder.php',
];
$backHref = $backMap[$typ] ?? 'index.php';

// Ausgebender (fix) + optionale Signaturdatei
$issuerName = 'Nils Bonk – Schriftführer';
$issuerSigRel = 'uploads/signatures/ausgebender.png';
$issuerSigAbs = __DIR__ . '/' . $issuerSigRel;
$hasIssuerSig = is_file($issuerSigAbs);
?>
<!doctype html>
<html lang="de">
<head>
  <meta charset="utf-8">
  <title><?= h($L['title']) ?></title>
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">

  <style>
    :root { --ink:#111; --muted:#6c757d; }
    body { color:var(--ink); background:#fff; }
    .doc { max-width: 900px; margin: 32px auto; padding: 24px 28px; background:#fff; border:1px solid #eee; border-radius:12px; }
    .doc h1 { font-size: 1.6rem; margin-bottom: .75rem; }
    .doc h2 { font-size: 1.15rem; color:var(--muted); margin-bottom: 1.25rem; }
    .kv th { width: 240px; white-space: nowrap; }
    .sigbox { display:flex; justify-content:space-between; align-items:flex-end; gap:32px; margin-top:56px; }
    .sig { flex:1; text-align:center; display:flex; flex-direction:column; justify-content:flex-end; min-height:140px; }
    .sig-line { margin-top:12px; border-top:1px solid #000; padding-top:6px; }
    .sig img.sig-img { max-height:70px; display:block; margin: 0 auto 8px; object-fit:contain; }
    .head-meta { display:flex; justify-content:space-between; align-items:flex-start; gap:16px; margin-bottom:16px; }
    .badge-e { background:#f1f3f5; border:1px solid #e9ecef; border-radius:999px; padding:.25rem .6rem; font-size:.85rem; }
    .printbar { position:sticky; top:0; background:#fff; border-bottom:1px solid #eee; padding:12px 0; margin-bottom:16px; }
    @media print {
      .no-print, .printbar { display:none !important; }
      .doc { border:none; border-radius:0; padding:0; margin:0; }
      body { background:#fff; }
    }
  </style>
</head>
<body>
  <div class="printbar no-print">
    <div class="container d-flex justify-content-between align-items-center">
      <div class="small text-muted">Druckansicht vorbereitet – <?= h($L['title']) ?></div>
      <div class="d-flex gap-2">
        <button onclick="window.print()" class="btn btn-primary">
          <i class="bi bi-printer"></i> Jetzt drucken
        </button>
        <a href="<?= h($backHref) ?>" class="btn btn-outline-secondary">Zurück</a>
      </div>
    </div>
  </div>

  <div class="doc">
    <div class="head-meta">
      <div>
        <h1 class="mb-0"><?= h($L['title']) ?></h1>
        <h2 class="mb-0">Einheit: <?= h($user_einheit) ?></h2>
      </div>
      <div class="text-end">
        <div><span class="badge-e">Datum:</span> <?= h($heute) ?></div>
      </div>
    </div>

    <table class="table table-sm kv">
      <tr><th>Kamerad</th>
          <td><?= h(trim(($k['vorname'] ?? '') . ' ' . ($k['nachname'] ?? ''))) ?: '– nicht zugeordnet –' ?></td></tr>
      <tr><th>Spindnummer</th><td><?= h($dat['spindnr'] ?? '—') ?></td></tr>
      <tr><th><?= h($L['field']) ?></th><td><strong><?= h($dat['nummer'] ?? '—') ?></strong></td></tr>
      <?php if ($typ === 'Transponder'): ?>
      <tr><th>Gebühr</th><td>10,00&nbsp;€</td></tr>
      <?php endif; ?>
      <?php if (!empty($dat['kommentar'])): ?>
      <tr><th>Kommentar</th><td><?= h($dat['kommentar']) ?></td></tr>
      <?php endif; ?>
    </table>

    <div class="mt-4" style="text-align:justify; line-height:1.6;">
      <?= $UE_TEXT ?>
    </div>

    <div class="sigbox">
      <div class="sig">
        <div class="sig-line">Unterschrift Empfänger</div>
        <div class="small text-muted">
          <?= h(trim(($k['vorname'] ?? '') . ' ' . ($k['nachname'] ?? ''))) ?: '—' ?>
        </div>
      </div>
      <div class="sig">
        <?php if ($hasIssuerSig): ?>
          <img class="sig-img" src="<?= h($issuerSigRel) ?>" alt="Unterschrift Ausgebender">
        <?php endif; ?>
        <div class="sig-line">Unterschrift Ausgebender</div>
        <div class="small text-muted"><?= h($issuerName) ?></div>
      </div>
    </div>

    <div class="mt-4">
      <div class="small text-muted">
        Hinweis: Bei Verlust ist unverzüglich die Wehrführung zu informieren. Unbefugte Weitergabe ist untersagt.
      </div>
    </div>
  </div>

  <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.css" rel="stylesheet">
</body>
</html>
