<?php
session_start();
require_once __DIR__ . '/includes/db.php';
require_once __DIR__ . '/includes/header.php';

if (!isset($_SESSION['user_id'])) {
    header("Location: login.php");
    exit;
}

$user_einheit   = $_SESSION['einheit'] ?? '';
$lieferschein_id = (int)($_GET['id'] ?? 0);

if ($lieferschein_id <= 0 || $user_einheit === '') {
    echo "<div class='container mt-5'><div class='alert alert-danger'>Ungültige Anfrage.</div></div>";
    include 'includes/footer.php';
    exit;
}

/* -------------------- Lieferschein laden (nur eigene Einheit) -------------------- */
$sqlL = "
    SELECT l.*,
           u.vorname, u.nachname,
           a.bezeichnung, a.strasse, a.plz, a.ort, a.hinweis
      FROM lieferscheine l
 LEFT JOIN users u   ON l.erstellt_von = u.id
 LEFT JOIN adressen a ON l.adresse_id = a.id
     WHERE l.id = ? AND l.einheit = ?
";
$stmt = $conn->prepare($sqlL);
$stmt->bind_param("is", $lieferschein_id, $user_einheit);
$stmt->execute();
$lieferschein = $stmt->get_result()->fetch_assoc();
$stmt->close();

if (!$lieferschein) {
    echo "<div class='container mt-5'><div class='alert alert-danger'>Lieferschein nicht gefunden oder keine Berechtigung.</div></div>";
    include 'includes/footer.php';
    exit;
}

/* -------------------- Geräte (mit Fahrzeug) laden -------------------- */
$sqlG = "
    SELECT g.bezeichnung,
           g.barcode,
           f.fahrzeugname
      FROM lieferschein_geraete lg
      JOIN ffw_geraete g         ON lg.geraet_id = g.id
 LEFT JOIN ffw_fahrzeuge f       ON g.fahrzeug = f.barcode_fzg
     WHERE lg.lieferschein_id = ?
  ORDER BY f.fahrzeugname ASC, g.bezeichnung ASC
";
$stG = $conn->prepare($sqlG);
$stG->bind_param("i", $lieferschein_id);
$stG->execute();
$geraete = $stG->get_result()->fetch_all(MYSQLI_ASSOC);
$stG->close();

/* -------------------- Freie Positionen laden -------------------- */
$sqlP = "
    SELECT bezeichnung, menge, bemerkung
      FROM lieferschein_positionen
     WHERE lieferschein_id = ?
  ORDER BY id ASC
";
$stP = $conn->prepare($sqlP);
$stP->bind_param("i", $lieferschein_id);
$stP->execute();
$positionen = $stP->get_result()->fetch_all(MYSQLI_ASSOC);
$stP->close();

/* -------------------- Helfer -------------------- */
function h($s){ return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8'); }

$anzGeraete   = count($geraete);
$anzPositionen= count($positionen);
$lieferdatum  = $lieferschein['lieferdatum'] ? date('d.m.Y', strtotime($lieferschein['lieferdatum'])) : '—';

$istGeliefert = (strtolower((string)$lieferschein['status']) === 'geliefert');
$rueckDatum   = (!empty($lieferschein['rueckmelde_datum'])) ? date('d.m.Y H:i', strtotime($lieferschein['rueckmelde_datum'])) : null;
?>
<style>
@media print {
  .no-print { display: none !important; }
  .page-break { page-break-before: always; }
  body { background: #fff !important; }
}

.page {
  max-width: 900px;
  margin: 30px auto;
  padding: 32px 36px;
  font-family: Arial, sans-serif;
  background: #fff;
  border: 1px solid #e5e7eb;
  border-radius: 10px;
}

.header-line {
  display: flex;
  justify-content: space-between;
  align-items: center;
  gap: 12px;
}

.brand {
  display: flex; align-items: center; gap: 14px;
}

.brand img { height: 72px; }
.brand h1 {
  font-size: 22px; margin: 0; font-weight: 700;
}

.meta {
  margin-top: 16px;
  display: grid;
  grid-template-columns: repeat(2, minmax(0,1fr));
  gap: 8px 24px;
  font-size: 14px;
}
.meta div strong { display: inline-block; min-width: 150px; }

h5 { margin-top: 22px; margin-bottom: 10px; font-size: 16px; }

.table {
  width: 100%;
  border-collapse: collapse;
  margin-top: 10px;
  page-break-inside: auto;
  font-size: 14px;
}
.table th, .table td {
  border: 1px solid #d1d5db;
  padding: 8px 10px;
  vertical-align: top;
}
.table thead th {
  background: #f3f4f6;
  font-weight: 700;
}
.table tfoot td {
  background: #f9fafb;
  font-weight: 600;
}

.badge {
  display: inline-block;
  padding: 2px 8px;
  border-radius: 999px;
  font-size: 12px;
  border: 1px solid #e5e7eb;
}
.badge-secondary { background: #f3f4f6; }
.badge-warning { background: #fff7ed; }
.badge-success { background: #ecfdf5; }

.address-card {
  border: 1px solid #e5e7eb;
  border-radius: 8px;
  padding: 10px 12px;
  background: #fafafa;
}

.actions { margin-top: 18px; }
.actions .btn { margin-right: 8px; }

.btn {
  display: inline-block;
  padding: 8px 12px;
  border-radius: 8px;
  border: 1px solid #d1d5db;
  text-decoration: none;
  font-size: 14px;
  background: #fff;
}
.btn-primary { background: #0d6efd; border-color: #0d6efd; color: #fff; }
.btn-secondary { background: #f3f4f6; }
</style>

<div class="page">
  <div class="header-line">
    <div class="brand">
      <img src="img/logo2.png" alt="Logo">
      <h1>Lieferschein</h1>
    </div>
    <div class="no-print">
      <a href="lieferschein_liste.php" class="btn btn-secondary">🔙 Zurück</a>
      <a href="javascript:window.print()" class="btn btn-primary">🖨️ Drucken</a>
    </div>
  </div>

  <div class="meta">
    <div><strong>Lieferscheinnummer:</strong> <?= (int)$lieferschein['id'] ?></div>
    <div><strong>Einheit:</strong> <?= h($lieferschein['einheit']) ?></div>
    <div><strong>Erstellt von:</strong> <?= h(trim(($lieferschein['vorname'] ?? '').' '.($lieferschein['nachname'] ?? ''))) ?></div>
    <div><strong>Lieferdatum:</strong> <?= $lieferdatum ?></div>
    <div>
      <strong>Status:</strong>
      <?php
        $st = strtolower((string)$lieferschein['status']);
        if ($st === 'offen')       echo '<span class="badge badge-secondary">offen</span>';
        elseif ($st === 'geliefert') echo '<span class="badge badge-success">geliefert</span>';
        else                       echo '<span class="badge badge-warning">'.h($lieferschein['status']).'</span>';
      ?>
    </div>
  </div>

  <h5>Empfänger</h5>
  <div class="address-card">
    <div><strong><?= h($lieferschein['empfaenger'] ?: ($lieferschein['bezeichnung'] ?? '')) ?></strong></div>
    <?php if (!empty($lieferschein['bezeichnung'])): ?>
      <div><?= h($lieferschein['bezeichnung']) ?></div>
    <?php endif; ?>
    <?php if (!empty($lieferschein['strasse'])): ?>
      <div><?= h($lieferschein['strasse']) ?></div>
    <?php endif; ?>
    <div><?= h(trim(($lieferschein['plz'] ?? '').' '.($lieferschein['ort'] ?? ''))) ?></div>
    <?php if (!empty($lieferschein['hinweis'])): ?>
      <div class="text-muted" style="margin-top:6px;"><em><?= nl2br(h($lieferschein['hinweis'])) ?></em></div>
    <?php endif; ?>
  </div>

  <h5 class="mt-4">📦 Geräte</h5>
  <table class="table">
    <thead>
      <tr>
        <th style="width:60px;">#</th>
        <th>Bezeichnung</th>
        <th>Barcode</th>
        <th>Fahrzeug</th>
      </tr>
    </thead>
    <tbody>
      <?php if ($anzGeraete === 0): ?>
        <tr><td colspan="4">Keine Geräte enthalten.</td></tr>
      <?php else: ?>
        <?php foreach ($geraete as $i => $g): ?>
          <tr>
            <td><?= $i + 1 ?></td>
            <td><?= h($g['bezeichnung']) ?></td>
            <td><?= h($g['barcode']) ?></td>
            <td><?= h($g['fahrzeugname'] ?: '-') ?></td>
          </tr>
        <?php endforeach; ?>
      <?php endif; ?>
    </tbody>
    <tfoot>
      <tr>
        <td colspan="3">Summe Geräte</td>
        <td><?= $anzGeraete ?></td>
      </tr>
    </tfoot>
  </table>

  <h5 class="mt-4">🧾 Freie Positionen</h5>
  <table class="table">
    <thead>
      <tr>
        <th style="width:60px;">#</th>
        <th>Bezeichnung</th>
        <th style="width:120px;">Menge</th>
        <th>Bemerkung</th>
      </tr>
    </thead>
    <tbody>
      <?php if ($anzPositionen === 0): ?>
        <tr><td colspan="4">Keine Positionen enthalten.</td></tr>
      <?php else: ?>
        <?php foreach ($positionen as $i => $p): ?>
          <tr>
            <td><?= $i + 1 ?></td>
            <td><?= h($p['bezeichnung']) ?></td>
            <td><?= h($p['menge']) ?></td>
            <td><?= h($p['bemerkung']) ?></td>
          </tr>
        <?php endforeach; ?>
      <?php endif; ?>
    </tbody>
    <tfoot>
      <tr>
        <td colspan="3">Summe Positionen</td>
        <td><?= $anzPositionen ?></td>
      </tr>
    </tfoot>
  </table>

  <?php if ($istGeliefert): ?>
    <h5 class="mt-4">✅ Rückmeldung</h5>
    <div class="address-card">
      <div><strong>Am:</strong> <?= $rueckDatum ?: '—' ?></div>
      <div style="margin-top:6px;"><strong>Info:</strong><br><?= nl2br(h($lieferschein['rueckmeldung'] ?? '')) ?></div>
    </div>
  <?php endif; ?>

  <div class="actions no-print">
    <a href="lieferschein_liste.php" class="btn btn-secondary">🔙 Zur Liste</a>
    <a href="javascript:window.print()" class="btn btn-primary">🖨️ Drucken</a>
  </div>
</div>
