<?php
// druck_melder.php – druckfertiges Übergabeprotokoll (Meldeempfänger)
ob_start();
session_start();

require_once __DIR__ . '/includes/db.php';

// Zugriffsschutz
if (!isset($_SESSION['user_id'])) { header('Location: login.php'); exit; }
$einheit = $_SESSION['einheit'] ?? '';
if ($einheit === '') { http_response_code(403); echo "Einheit fehlt."; exit; }

// Eingaben
$id  = (int)($_GET['id'] ?? 0);
if ($id <= 0) { http_response_code(400); echo "Ungültige ID."; exit; }

// Melder laden (nur eigene Einheit)
$stmt = $conn->prepare("
  SELECT m.*
  FROM melder m
  WHERE m.id = ? AND m.einheit = ?
  LIMIT 1
");
$stmt->bind_param('is', $id, $einheit);
$stmt->execute();
$melder = $stmt->get_result()->fetch_assoc();
if (!$melder) { http_response_code(404); echo "Melder nicht gefunden."; exit; }

// Kamerad aus kamerad_id (falls zugeordnet)
$k = null;
if (!empty($melder['kamerad_id'])) {
  $ks = $conn->prepare("
    SELECT vorname, nachname
    FROM kameraden
    WHERE id = ? AND einheit = ?
    LIMIT 1
  ");
  $kid = (int)$melder['kamerad_id'];
  $ks->bind_param('is', $kid, $einheit);
  $ks->execute();
  $k = $ks->get_result()->fetch_assoc();
}

function h($s){ return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8'); }

// Labels
$title = 'Übergabeprotokoll – Meldeempfänger';

// Übergabetext (allgemein; bei Bedarf später spezifisch ersetzen)
$UE_TEXT = "<p>Hiermit wird die Übergabe des oben genannten Objekts bestätigt. "
         . "Der Empfänger verpflichtet sich zur sorgfältigen Aufbewahrung, zweckgebundenen Nutzung sowie "
         . "zur unverzüglichen Meldung von Verlust/Beschädigung und zur Rückgabe auf Aufforderung oder bei Ausscheiden.</p>";

// Heutiges Datum
$heute = (new DateTime('now'))->format('d.m.Y');

// Rück-Link
$backHref = 'melder.php';

// Ausgebender (fix) + optionale Signaturdatei
$issuerName  = 'Nils Bonk – Schriftführer';
$issuerSigRel = 'uploads/signatures/ausgebender.png'; // Web-Pfad
$issuerSigAbs = __DIR__ . '/' . $issuerSigRel;        // Server-Pfad
$hasIssuerSig = is_file($issuerSigAbs);

// Zubehör-String
$zube = [];
if (!empty($melder['ladegeraet']))   $zube[] = 'Ladegerät';
if (!empty($melder['antenne']))      $zube[] = 'Antenne';
if (!empty($melder['schutztasche'])) $zube[] = 'Schutztasche';
$zubeText = empty($zube) ? '—' : implode(', ', $zube);
?>
<!doctype html>
<html lang="de">
<head>
  <meta charset="utf-8">
  <title><?= h($title) ?></title>
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">

  <style>
    :root { --ink:#111; --muted:#6c757d; }
    body { color:var(--ink); background:#fff; }
    .doc { max-width: 900px; margin: 32px auto; padding: 24px 28px; background:#fff; border:1px solid #eee; border-radius:12px; }
    .doc h1 { font-size: 1.6rem; margin-bottom: .75rem; }
    .doc h2 { font-size: 1.15rem; color:var(--muted); margin-bottom: 1.25rem; }
    .kv th { width: 240px; white-space: nowrap; }
    .sigbox { display:flex; justify-content:space-between; align-items:flex-end; gap:32px; margin-top:56px; }
    .sig { flex:1; text-align:center; display:flex; flex-direction:column; justify-content:flex-end; min-height:140px; }
    .sig-line { margin-top:12px; border-top:1px solid #000; padding-top:6px; }
    .sig img.sig-img { max-height:70px; display:block; margin: 0 auto 8px; object-fit:contain; }
    .head-meta { display:flex; justify-content:space-between; align-items:flex-start; gap:16px; margin-bottom:16px; }
    .badge-e { background:#f1f3f5; border:1px solid #e9ecef; border-radius:999px; padding:.25rem .6rem; font-size:.85rem; }
    .printbar { position:sticky; top:0; background:#fff; border-bottom:1px solid #eee; padding:12px 0; margin-bottom:16px; }
    @media print {
      .no-print, .printbar { display:none !important; }
      .doc { border:none; border-radius:0; padding:0; margin:0; }
      body { background:#fff; }
    }
  </style>
</head>
<body>
  <div class="printbar no-print">
    <div class="container d-flex justify-content-between align-items-center">
      <div class="small text-muted">Druckansicht vorbereitet – <?= h($title) ?></div>
      <div class="d-flex gap-2">
        <button onclick="window.print()" class="btn btn-primary">
          <i class="bi bi-printer"></i> Jetzt drucken
        </button>
        <a href="<?= h($backHref) ?>" class="btn btn-outline-secondary">Zurück</a>
      </div>
    </div>
  </div>

  <div class="doc">
    <div class="head-meta">
      <div>
        <h1 class="mb-0"><?= h($title) ?></h1>
        <h2 class="mb-0">Einheit: <?= h($einheit) ?></h2>
      </div>
      <div class="text-end">
        <div><span class="badge-e">Datum:</span> <?= h($heute) ?></div>
      </div>
    </div>

    <table class="table table-sm kv">
      <tr><th>Kamerad</th>
          <td><?= h(trim(($k['vorname'] ?? '') . ' ' . ($k['nachname'] ?? ''))) ?: '– nicht zugeordnet –' ?></td></tr>
      <tr><th>Melder-Kennung</th><td><strong><?= h($melder['kennung'] ?? '—') ?></strong></td></tr>
      <tr><th>Seriennummer</th><td><?= h($melder['seriennummer'] ?? '—') ?></td></tr>
      <tr><th>Typ</th><td><?= h($melder['typ'] ?? '—') ?></td></tr>
      <tr><th>Zubehör</th><td><?= h($zubeText) ?></td></tr>
      <?php if (!empty($melder['ausgabe_datum'])): ?>
      <tr><th>Ausgabe-Datum</th><td><?= h(date('d.m.Y', strtotime($melder['ausgabe_datum']))) ?></td></tr>
      <?php endif; ?>
    </table>

    <div class="mt-4" style="text-align:justify; line-height:1.6;">
      <?= $UE_TEXT /* HTML bewusst direkt ausgeben */ ?>
    </div>

    <div class="sigbox">
      <div class="sig">
        <div class="sig-line">Unterschrift Empfänger</div>
        <div class="small text-muted">
          <?= h(trim(($k['vorname'] ?? '') . ' ' . ($k['nachname'] ?? ''))) ?: '—' ?>
        </div>
      </div>
      <div class="sig">
        <?php if ($hasIssuerSig): ?>
          <img class="sig-img" src="<?= h($issuerSigRel) ?>" alt="Unterschrift Ausgebender">
        <?php endif; ?>
        <div class="sig-line">Unterschrift Ausgebender</div>
        <div class="small text-muted"><?= h($issuerName) ?></div>
      </div>
    </div>

    <div class="mt-4">
      <div class="small text-muted">
        Hinweis: Bei Verlust ist unverzüglich die Wehrführung zu informieren. Unbefugte Weitergabe ist untersagt.
      </div>
    </div>
  </div>

  <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.css" rel="stylesheet">
</body>
</html>
