<?php
// lager_buchung.php – Lagerbuchungen (einheitsgebunden, sauber & sicher)
ob_start();
session_start();

require_once __DIR__ . '/includes/db.php';
require_once __DIR__ . '/includes/header.php';
require_once __DIR__ . '/includes/nav.php';

if (!isset($_SESSION['user_id'])) {
    header("Location: login.php");
    exit;
}

$user_id       = (int)$_SESSION['user_id'];
$user_einheit  = $_SESSION['einheit'] ?? '';
$meldung       = '';

// CSRF-Token
if (empty($_SESSION['csrf'])) {
  $_SESSION['csrf'] = bin2hex(random_bytes(16));
}
$CSRF = $_SESSION['csrf'];

// Artikel-ID aus GET (0 = Suchmaske)
$id = (int)($_GET['id'] ?? 0);

// --- Formular speichern, falls ausgefüllt ---
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['buchung_speichern']) && isset($_POST['artikel_id'])) {
    // CSRF
    if (!hash_equals($_SESSION['csrf'] ?? '', $_POST['csrf'] ?? '')) {
        $meldung = "<div class='alert alert-danger'>Sicherheitsprüfung fehlgeschlagen.</div>";
    } else {
        $id          = (int)$_POST['artikel_id'];
        $buchungstyp = $_POST['buchungstyp'] === 'Zugang' ? 'Zugang' : 'Abgang';
        $anzahl      = max(0, (int)$_POST['anzahl']);
        $kamerad_id  = ($_POST['kamerad_id'] !== '' ? (int)$_POST['kamerad_id'] : null);
        $bemerkung   = trim($_POST['bemerkung'] ?? '');

        if ($anzahl > 0 && $id > 0 && $user_einheit !== '') {
            // Artikel der eigenen Einheit laden (inkl. aktuellem Bestand)
            $st = $conn->prepare("SELECT id, bestand FROM lager_bestand WHERE id = ? AND einheit = ?");
            $st->bind_param("is", $id, $user_einheit);
            $st->execute();
            $artikelRow = $st->get_result()->fetch_assoc();
            $st->close();

            if (!$artikelRow) {
                $meldung = "<div class='alert alert-danger'>Artikel nicht gefunden oder keine Berechtigung.</div>";
            } else {
                $bestand_alt = (int)$artikelRow['bestand'];
                $bestand_neu = ($buchungstyp === 'Zugang') ? ($bestand_alt + $anzahl) : ($bestand_alt - $anzahl);

                // Negative Bestände verhindern
                if ($bestand_neu < 0) {
                    $meldung = "<div class='alert alert-danger'>Abgang zu hoch – Bestand darf nicht negativ werden.</div>";
                } else {
                    // Bestand aktualisieren (nur eigene Einheit)
                    $stmt = $conn->prepare("UPDATE lager_bestand SET bestand = ? WHERE id = ? AND einheit = ?");
                    $stmt->bind_param("iis", $bestand_neu, $id, $user_einheit);
                    $okUpd = $stmt->execute();
                    $stmt->close();

                    if ($okUpd && $conn->affected_rows >= 0) {
                        // Historie schreiben (mit Datum)
                        // lager_historie: id, lager_id, buchungstyp, anzahl, datum, kamerad_id, bemerkung
                        if ($kamerad_id) {
                            $stmt_hist = $conn->prepare("INSERT INTO lager_historie (lager_id, buchungstyp, anzahl, datum, kamerad_id, bemerkung) VALUES (?, ?, ?, NOW(), ?, ?)");
                            $stmt_hist->bind_param("isiss", $id, $buchungstyp, $anzahl, $kamerad_id, $bemerkung);
                        } else {
                            $null = null;
                            $stmt_hist = $conn->prepare("INSERT INTO lager_historie (lager_id, buchungstyp, anzahl, datum, kamerad_id, bemerkung) VALUES (?, ?, ?, NOW(), NULL, ?)");
                            $stmt_hist->bind_param("isss", $id, $buchungstyp, $anzahl, $bemerkung);
                        }
                        $stmt_hist->execute();
                        $stmt_hist->close();

                        $meldung = "<div class='alert alert-success'>Buchung erfolgreich gespeichert. Neuer Bestand: <strong>{$bestand_neu}</strong></div>";

                        // Nach Buchung zurück zur Suche
                        $id = 0;
                    } else {
                        $meldung = "<div class='alert alert-danger'>Aktualisierung fehlgeschlagen.</div>";
                    }
                }
            }
        } else {
            $meldung = "<div class='alert alert-danger'>Ungültige Eingaben.</div>";
        }
    }
}
?>
<div class="container my-5">
  <h2 class="mb-4">Lager – Buchen</h2>

  <?= $meldung ?>

  <?php if ($id === 0): ?>
    <!-- Artikelsuche (AJAX) – Ergebnis klickbar, springt in dieses Formular -->
    <div class="card p-4 bg-white shadow-sm">
      <div class="mb-3">
        <label class="form-label">Artikelsuche</label>
        <input type="text" id="artikelSuchfeld" class="form-control" placeholder="Artikelnummer oder Name eingeben...">
        <div class="form-text">Suche ist auf deine Einheit eingeschränkt.</div>
      </div>

      <ul class="list-group" id="suchErgebnisse" style="display:none;"></ul>

      <div class="mt-4 d-flex gap-2">
        <a href="lager.php" class="btn btn-secondary">&laquo; Zurück</a>
      </div>
    </div>

    <script>
    document.addEventListener('DOMContentLoaded', function() {
      const suchfeld = document.getElementById('artikelSuchfeld');
      const ergebnisListe = document.getElementById('suchErgebnisse');

      let lastController = null;

      suchfeld.addEventListener('input', function() {
        const suchbegriff = suchfeld.value.trim();

        if (suchbegriff.length < 2) {
          ergebnisListe.style.display = 'none';
          ergebnisListe.innerHTML = '';
          return;
        }

        // Abort vorheriger Request
        if (lastController) lastController.abort();
        lastController = new AbortController();

        fetch('lager_suche.php?s=' + encodeURIComponent(suchbegriff), { signal: lastController.signal })
          .then(response => response.json())
          .then(daten => {
            ergebnisListe.innerHTML = '';

            if (Array.isArray(daten) && daten.length > 0) {
              daten.forEach(artikel => {
                const li = document.createElement('li');
                li.classList.add('list-group-item', 'list-group-item-action', 'd-flex', 'justify-content-between', 'align-items-center');
                li.innerHTML = `
                  <div>
                    <div class="fw-semibold">${artikel.artikelnummer}</div>
                    <div class="text-muted small">${artikel.artikelname}</div>
                  </div>
                  <span class="badge text-bg-light">${artikel.bestand ?? 0}</span>
                `;
                li.onclick = function() {
                  window.location.href = 'lager_buchung.php?id=' + artikel.id;
                };
                ergebnisListe.appendChild(li);
              });
              ergebnisListe.style.display = 'block';
            } else {
              ergebnisListe.style.display = 'none';
            }
          })
          .catch(()=>{ /* ignorieren (abort) */ });
      });
    });
    </script>

  <?php else: ?>
    <?php
    // Artikel der eigenen Einheit laden
    $stmt = $conn->prepare("SELECT * FROM lager_bestand WHERE id = ? AND einheit = ?");
    $stmt->bind_param("is", $id, $user_einheit);
    $stmt->execute();
    $artikel = $stmt->get_result()->fetch_assoc();
    $stmt->close();

    if (!$artikel) {
        echo "<div class='alert alert-danger'>Artikel nicht gefunden oder keine Berechtigung.</div>";
    } else {
        // Kameraden nur aus eigener Einheit
        $stmtK = $conn->prepare("SELECT id, vorname, nachname FROM kameraden WHERE einheit = ? ORDER BY nachname ASC, vorname ASC");
        $stmtK->bind_param("s", $user_einheit);
        $stmtK->execute();
        $kameraden = $stmtK->get_result();
        $stmtK->close();

        $warnung = ((int)$artikel['bestand'] <= (int)$artikel['mindesbestand']);
    ?>
    <div class="card p-4 bg-white shadow-sm">
      <h4 class="mb-4">
        Buchung für:
        <span class="text-monospace"><?= htmlspecialchars($artikel['artikelnummer']) ?></span>
        – <?= htmlspecialchars($artikel['artikelname']) ?>
      </h4>

      <div class="row g-3 mb-2">
        <div class="col-6 col-md-3">
          <div class="border rounded p-2">
            <div class="text-muted small">Kategorie</div>
            <div class="fw-semibold"><?= htmlspecialchars($artikel['kategorie'] ?: '–') ?></div>
          </div>
        </div>
        <div class="col-6 col-md-3">
          <div class="border rounded p-2">
            <div class="text-muted small">Größe</div>
            <div class="fw-semibold"><?= htmlspecialchars($artikel['groesse'] ?: '–') ?></div>
          </div>
        </div>
        <div class="col-6 col-md-3">
          <div class="border rounded p-2">
            <div class="text-muted small">Bestand</div>
            <div class="fw-semibold"><?= (int)$artikel['bestand'] ?></div>
          </div>
        </div>
        <div class="col-6 col-md-3">
          <div class="border rounded p-2">
            <div class="text-muted small">Mindestbestand</div>
            <div class="fw-semibold">
              <?= (int)$artikel['mindesbestand'] ?>
              <?php if ($warnung): ?>
                <span class="badge bg-danger ms-2">Unterschritten!</span>
              <?php endif; ?>
            </div>
          </div>
        </div>
      </div>

      <form method="post" class="mt-3">
        <input type="hidden" name="csrf" value="<?= htmlspecialchars($CSRF) ?>">
        <input type="hidden" name="artikel_id" value="<?= $id ?>">

        <div class="row g-3">
          <div class="col-md-4">
            <label class="form-label">Buchungstyp</label>
            <select name="buchungstyp" class="form-select" required>
              <option value="Zugang">Zugang (+)</option>
              <option value="Abgang">Abgang (-)</option>
            </select>
          </div>

          <div class="col-md-4">
            <label class="form-label">Anzahl</label>
            <input type="number" name="anzahl" class="form-control" min="1" required>
          </div>

          <div class="col-md-4">
            <label class="form-label">Kamerad (optional)</label>
            <select name="kamerad_id" class="form-select">
              <option value="">– Keine Auswahl –</option>
              <?php while ($k = $kameraden->fetch_assoc()): ?>
                <option value="<?= (int)$k['id'] ?>"><?= htmlspecialchars($k['nachname'] . ', ' . $k['vorname']) ?></option>
              <?php endwhile; ?>
            </select>
          </div>

          <div class="col-12">
            <label class="form-label">Bemerkung (optional)</label>
            <textarea name="bemerkung" class="form-control" rows="2" placeholder="z. B. Ausgabe an Kamerad, Grund der Korrektur …"></textarea>
          </div>
        </div>

        <div class="d-flex gap-2 mt-4">
          <button type="submit" name="buchung_speichern" class="btn btn-primary">💾 Buchung speichern</button>
          <a href="lager_buchung.php" class="btn btn-secondary">Anderen Artikel buchen</a>
          <a href="lager_historie.php?id=<?= (int)$artikel['id'] ?>" class="btn btn-outline-secondary">Historie ansehen</a>
        </div>
      </form>
    </div>
    <?php } ?>
  <?php endif; ?>
</div>

<?php include 'includes/footer.php'; ob_end_flush(); ?>
