<?php
// geraete_papierkorb.php – Ausgesonderte Geräte (Papierkorb)
ob_start();
session_start();

require_once __DIR__ . '/includes/db.php';
require_once __DIR__ . '/includes/header.php';
require_once __DIR__ . '/includes/nav.php';

if (!isset($_SESSION['user_id'])) {
    echo "<div class='container my-5'><div class='alert alert-danger'>Zugriff verweigert.</div></div>";
    include 'includes/footer.php'; exit;
}

$einheit = $_SESSION['einheit'] ?? '';
if ($einheit === '') {
    echo "<div class='container my-5'><div class='alert alert-warning'>⚠️ Keine Einheit in der Session gefunden. Bitte erneut einloggen.</div></div>";
    include 'includes/footer.php'; exit;
}

// --- Helpers ---
function h($s){ return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8'); }
function fmtDate($v){
  // lpruefung wird oft als Unix-Timestamp (int) gespeichert – robust abfedern:
  if ($v === null || $v === '' ) return '—';
  if (is_numeric($v)) {
    $iv = (int)$v; if ($iv <= 0) return '—';
    return date('d.m.Y', $iv);
  }
  // Falls als Y-m-d gespeichert
  $ts = strtotime($v);
  return $ts ? date('d.m.Y', $ts) : '—';
}

// CSRF
if (empty($_SESSION['csrf'])) { $_SESSION['csrf'] = bin2hex(random_bytes(16)); }
$CSRF = $_SESSION['csrf'];

$meldung = '';

// --- Aktionen: Wiederherstellen / Löschen ---
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    if (!hash_equals($_SESSION['csrf'] ?? '', $_POST['csrf'] ?? '')) {
        $meldung = "<div class='alert alert-danger'>Sicherheitsprüfung fehlgeschlagen.</div>";
    } else {
        $id = (int)($_POST['id'] ?? 0);
        if ($id > 0) {
            if ($action === 'restore') {
                // Status zurück auf 'frei'
                $st = $conn->prepare("UPDATE ffw_geraete SET status='frei' WHERE id=? AND einheit=? AND LOWER(status)='ausgesondert'");
                $st->bind_param("is", $id, $einheit);
                $st->execute();
                $meldung = ($st->affected_rows > 0)
                  ? "<div class='alert alert-success'>♻️ Gerät wiederhergestellt (Status: frei).</div>"
                  : "<div class='alert alert-info'>Kein passender Datensatz gefunden oder bereits aktiv.</div>";
            } elseif ($action === 'purge') {
                // Endgültig löschen
                $st = $conn->prepare("DELETE FROM ffw_geraete WHERE id=? AND einheit=? AND LOWER(status)='ausgesondert'");
                $st->bind_param("is", $id, $einheit);
                $st->execute();
                $meldung = ($st->affected_rows > 0)
                  ? "<div class='alert alert-success'>🗑️ Gerät endgültig gelöscht.</div>"
                  : "<div class='alert alert-info'>Kein passender Datensatz gefunden.</div>";
            }
        }
    }
}

// --- Filter & Suche ---
$search   = trim($_GET['search'] ?? '');
$f_raum   = trim($_GET['gr'] ?? '');
$f_fahrzg = trim($_GET['fahrzeug'] ?? '');

$where  = "LOWER(status)='ausgesondert' AND einheit=?";
$types  = "s";
$params = [$einheit];

if ($search !== '') {
  $like = '%'.$search.'%';
  $where .= " AND (bezeichnung LIKE ? OR barcode LIKE ? OR kat LIKE ? OR gr LIKE ? OR fahrzeug LIKE ? OR standort LIKE ? OR memo LIKE ?)";
  for($i=0;$i<7;$i++){ $params[] = $like; $types .= 's'; }
}
if ($f_raum !== '') {
  $where .= " AND gr = ?";
  $params[] = $f_raum; $types .= 's';
}
if ($f_fahrzg !== '') {
  $where .= " AND fahrzeug = ?";
  $params[] = $f_fahrzg; $types .= 's';
}

// --- Option: CSV-Export ---
if (isset($_GET['export']) && $_GET['export'] === 'csv') {
  $fn = 'geraete_papierkorb_' . date('Y-m-d') . '.csv';
  header('Content-Type: text/csv; charset=utf-8');
  header("Content-Disposition: attachment; filename=\"$fn\"");
  $out = fopen('php://output', 'w');
  fputcsv($out, ['Bezeichnung','Barcode','Fahrzeug','Geräteraum','Kategorie','Baujahr','LetztePrüfung','Standort','Notiz']);

  $sqlE = "SELECT bezeichnung, barcode, fahrzeug, gr, kat, baujahr, lpruefung, standort, memo
           FROM ffw_geraete WHERE $where ORDER BY bezeichnung ASC";
  $stE = $conn->prepare($sqlE);
  $a = [$types]; foreach($params as $k=>$_){ $a[] = &$params[$k]; }
  call_user_func_array([$stE,'bind_param'],$a);
  $stE->execute();
  $resE = $stE->get_result();
  while ($r = $resE->fetch_assoc()) {
    fputcsv($out, [
      $r['bezeichnung'],
      $r['barcode'],
      $r['fahrzeug'],
      $r['gr'],
      $r['kat'],
      $r['baujahr'],
      fmtDate($r['lpruefung']),
      $r['standort'],
      $r['memo']
    ]);
  }
  fclose($out); exit;
}

// --- Dropdown-Daten (für Filter) ---
$raeume = $conn->prepare("SELECT DISTINCT gr FROM ffw_geraete WHERE einheit=? AND LOWER(status)='ausgesondert' AND gr<>'' ORDER BY gr ASC");
$raeume->bind_param("s", $einheit); $raeume->execute(); $rooms = $raeume->get_result();

$fahrzeuge = $conn->prepare("SELECT DISTINCT fahrzeug FROM ffw_geraete WHERE einheit=? AND LOWER(status)='ausgesondert' AND fahrzeug<>'' ORDER BY fahrzeug ASC");
$fahrzeuge->bind_param("s", $einheit); $fahrzeuge->execute(); $cars = $fahrzeuge->get_result();

// --- KPIs ---
$sqlKpi = "SELECT
            COUNT(*)                                                         AS cnt,
            SUM(CASE WHEN gr<>'' THEN 1 ELSE 0 END)                          AS mit_raum,
            SUM(CASE WHEN fahrzeug<>'' THEN 1 ELSE 0 END)                    AS mit_fahrzeug
          FROM ffw_geraete WHERE $where";
$stK = $conn->prepare($sqlKpi);
$a = [$types]; foreach($params as $k=>$_){ $a[] = &$params[$k]; }
call_user_func_array([$stK,'bind_param'],$a);
$stK->execute();
$K = $stK->get_result()->fetch_assoc() ?: ['cnt'=>0,'mit_raum'=>0,'mit_fahrzeug'=>0];

// --- Liste laden ---
$sql = "SELECT * FROM ffw_geraete WHERE $where ORDER BY bezeichnung ASC";
$st = $conn->prepare($sql);
$a = [$types]; foreach($params as $k=>$_){ $a[] = &$params[$k]; }
call_user_func_array([$st,'bind_param'],$a);
$st->execute();
$result = $st->get_result();
?>
<style>
  .kpi-card{border:1px solid #eee;border-radius:14px;padding:16px;background:#fff}
  .kpi-val{font-size:1.6rem;font-weight:700}
  .kpi-sub{font-size:.9rem;color:#6c757d}
</style>

<div class="container my-5">
  <div class="d-flex flex-wrap justify-content-between align-items-end gap-2 mb-3">
    <div>
      <h2 class="mb-1 text-danger">🗑️ Ausgesonderte Geräte (Papierkorb)</h2>
      <div class="text-muted small">Einheit: <span class="badge text-bg-secondary"><?= h($einheit) ?></span></div>
    </div>

    <form method="get" class="d-flex flex-wrap gap-2">
      <input type="text" name="search" class="form-control" placeholder="🔍 Suche: Bezeichnung/Barcode/Kategorie/Raum/Standort/Notiz" value="<?= h($search) ?>">

      <select name="gr" class="form-select">
        <option value="">Alle Geräteräume</option>
        <?php while($r = $rooms->fetch_assoc()): ?>
          <option value="<?= h($r['gr']) ?>" <?= $f_raum===$r['gr']?'selected':'' ?>><?= h($r['gr']) ?></option>
        <?php endwhile; ?>
      </select>

      <select name="fahrzeug" class="form-select">
        <option value="">Alle Fahrzeuge</option>
        <?php while($f = $cars->fetch_assoc()): ?>
          <option value="<?= h($f['fahrzeug']) ?>" <?= $f_fahrzg===$f['fahrzeug']?'selected':'' ?>><?= h($f['fahrzeug']) ?></option>
        <?php endwhile; ?>
      </select>

      <button class="btn btn-outline-primary">Anwenden</button>
      <?php if ($search!=='' || $f_raum!=='' || $f_fahrzg!==''): ?>
        <a href="geraete_papierkorb.php" class="btn btn-outline-secondary">Zurücksetzen</a>
      <?php endif; ?>

      <a href="?export=csv&<?= http_build_query(['search'=>$search,'gr'=>$f_raum,'fahrzeug'=>$f_fahrzg]) ?>" class="btn btn-outline-secondary">CSV Export</a>
    </form>
  </div>

  <?= $meldung ?>

  <!-- KPIs -->
  <div class="row g-3 mb-4">
    <div class="col-6 col-md-3"><div class="kpi-card"><div class="kpi-sub">Gesamt</div><div class="kpi-val"><?= (int)$K['cnt'] ?></div></div></div>
    <div class="col-6 col-md-3"><div class="kpi-card"><div class="kpi-sub">mit Geräteraum</div><div class="kpi-val"><?= (int)$K['mit_raum'] ?></div></div></div>
    <div class="col-6 col-md-3"><div class="kpi-card"><div class="kpi-sub">mit Fahrzeug</div><div class="kpi-val"><?= (int)$K['mit_fahrzeug'] ?></div></div></div>
  </div>

  <div class="card shadow-sm">
    <div class="table-responsive">
      <table class="table table-hover align-middle mb-0">
        <thead class="table-danger">
          <tr>
            <th>Bezeichnung</th>
            <th>Barcode</th>
            <th>Fahrzeug</th>
            <th>Geräteraum</th>
            <th>Kategorie</th>
            <th>Baujahr</th>
            <th>Letzte Prüfung</th>
            <th>Standort</th>
            <th>Notiz</th>
            <th class="text-end" style="width:60px;"></th>
          </tr>
        </thead>
        <tbody>
        <?php if ($result->num_rows === 0): ?>
          <tr><td colspan="10" class="text-center text-muted py-4">Keine ausgesonderten Geräte gefunden.</td></tr>
        <?php else: while ($row = $result->fetch_assoc()): ?>
          <tr>
            <td>
              <div class="fw-semibold"><?= h($row['bezeichnung']) ?></div>
              <div class="text-muted small"><?= h($row['kat'] ?: '—') ?></div>
            </td>
            <td class="text-monospace"><?= h($row['barcode']) ?></td>
            <td><?= h($row['fahrzeug'] ?: '—') ?></td>
            <td><?= h($row['gr'] ?: '—') ?></td>
            <td><?= h($row['kat'] ?: '—') ?></td>
            <td><?= h($row['baujahr'] ?: '—') ?></td>
            <td><?= fmtDate($row['lpruefung']) ?></td>
            <td><?= h($row['standort'] ?: '—') ?></td>
            <td><?= nl2br(h($row['memo'] ?: '')) ?></td>
            <td class="text-end">
              <div class="dropdown">
                <button class="btn btn-sm btn-outline-secondary dropdown-toggle" data-bs-toggle="dropdown">Aktionen</button>
                <div class="dropdown-menu dropdown-menu-end">
                  <form method="post" class="px-3 py-1"
                        onsubmit="return confirm('Gerät wiederherstellen? Status wird auf „frei“ gesetzt.');">
                    <input type="hidden" name="csrf" value="<?= h($CSRF) ?>">
                    <input type="hidden" name="id" value="<?= (int)$row['id'] ?>">
                    <input type="hidden" name="action" value="restore">
                    <button class="dropdown-item">♻️ Wiederherstellen</button>
                  </form>
                  <div class="dropdown-divider"></div>
                  <form method="post" class="px-3 py-1"
                        onsubmit="return confirm('Endgültig löschen? Dieser Vorgang kann nicht rückgängig gemacht werden.');">
                    <input type="hidden" name="csrf" value="<?= h($CSRF) ?>">
                    <input type="hidden" name="id" value="<?= (int)$row['id'] ?>">
                    <input type="hidden" name="action" value="purge">
                    <button class="dropdown-item text-danger">🗑️ Endgültig löschen</button>
                  </form>
                </div>
              </div>
            </td>
          </tr>
        <?php endwhile; endif; ?>
        </tbody>
      </table>
    </div>
  </div>
</div>

<?php include 'includes/footer.php'; ob_end_flush(); ?>
